<?php
class Node
{
    public $data;
    public $left;
    public $right;
    public $height;

    public function __construct($data) {
        $this->data = $data;
        $this->left = null;
        $this->right = null;
        $this->height = 1;
    }
}

class AVLTree
{
    public $root;

    // Helper function to get the height of a node
    private function height($node) {
        if ($node === null) {
            return 0;
        }
        return $node->height;
    }

    // Helper function to calculate the balancing factor of a node
    private function balanceFactor($node) {
        return $this->height($node->left) - $this->height($node->right);
    }

    // Function to perform rotations to maintain the balance of the tree
    private function balance($node) {
        if ($node === null) {
            return null;
        }

        $node->height = 1 + max($this->height($node->left), $this->height($node->right));

        $balance = $this->balanceFactor($node);

        // Rotation cases
        if ($balance > 1) {
            if ($this->balanceFactor($node->left) >= 0) {
                return $this->rotateRight($node);
            } else {
                $node->left = $this->rotateLeft($node->left);
                return $this->rotateRight($node);
            }
        }

        if ($balance < -1) {
            if ($this->balanceFactor($node->right) <= 0) {
                return $this->rotateLeft($node);
            } else {
                $node->right = $this->rotateRight($node->right);
                return $this->rotateLeft($node);
            }
        }

        return $node;
    }

    // Rotation Functions
    private function rotateRight($y)
    {
        $x = $y->left;
        $T2 = $x->right;

        $x->right = $y;
        $y->left = $T2;

        $y->height = 1 + max($this->height($y->left), $this->height($y->right));
        $x->height = 1 + max($this->height($x->left), $this->height($x->right));

        return $x;
    }

    private function rotateLeft($x)
    {
        $y = $x->right;
        $T2 = $y->left;

        $y->left = $x;
        $x->right = $T2;

        $x->height = 1 + max($this->height($x->left), $this->height($x->right));
        $y->height = 1 + max($this->height($y->left), $this->height($y->right));

        return $y;
    }

    // Insert function
    public function insert($root, $data)
    {
        if ($root === null)
        {
            return new Node($data);
        }

        if ($data < $root->data)
        {
            $root->left = $this->insert($root->left, $data);
        } elseif ($data > $root->data)
        {
            $root->right = $this->insert($root->right, $data);
        } else
        {
            // Duplicates are not allowed
            return $root;
        }

        return $this->balance($root);
    }

    // Removal function
    public function remove($root, $data)
    {
        if ($root === null)
        {
            return $root;
        }

        if ($data < $root->data)
        {
            $root->left = $this->remove($root->left, $data);
        } elseif ($data > $root->data)
        {
            $root->right = $this->remove($root->right, $data);
        } else {
            // Node found, perform removal

            // Case 1: Node with one child or no children
            if ($root->left === null || $root->right === null)
            {
                $temp = ($root->left !== null) ? $root->left : $root->right;

                // Case without child
                if ($temp === null)
                {
                    $temp = $root;
                    $root = null;
                } else
                { // Affair with a son
                    $root = $temp; // Copies the contents of the non-null child
                }
                unset($temp);
            } else
            {
                // Case 2: Node with two children, find in-order successor (smallest node in right subtree)
                $temp = $this->minValueNode($root->right);
                // Copies the contents of the in-order successor to this node
                $root->data = $temp->data;
                // Remove o sucessor in-order
                $root->right = $this->remove($root->right, $temp->data);
            }
        }

        if ($root === null)
        {
            return $root;
        }

        return $this->balance($root);
    }

    // Helper function to find the node with the smallest value in the tree
    private function minValueNode($node)
    {
        $current = $node;
        while ($current->left !== null)
        {
            $current = $current->left;
        }
        return $current;
    }

    // Public insert function
    public function insertData($data)
    {
        $this->root = $this->insert($this->root, $data);
    }

    // Public removal function
    public function removeData($data)
    {
        $this->root = $this->remove($this->root, $data);
    }

    // Function to perform an orderly traversal of the tree
    public function inOrderTraversal($node)
    {
        if ($node !== null)
        {
            $this->inOrderTraversal($node->left);
            echo $node->data . " ";
            $this->inOrderTraversal($node->right);
        }
    }
}